<?php
/**
 * Amir Ahmed Theme Functions
 *
 * @package Amir_Ahmed
 * @version 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Theme Setup
 */
function amir_ahmed_setup() {
    // Add theme support
    add_theme_support('title-tag');
    add_theme_support('post-thumbnails');
    add_theme_support('custom-logo', array(
        'height'      => 100,
        'width'       => 400,
        'flex-height' => true,
        'flex-width'  => true,
    ));
    add_theme_support('html5', array(
        'search-form',
        'comment-form',
        'comment-list',
        'gallery',
        'caption',
        'style',
        'script',
    ));
    add_theme_support('customize-selective-refresh-widgets');
    add_theme_support('editor-styles');
    add_theme_support('responsive-embeds');

    // Add image sizes
    add_image_size('portfolio-thumb', 600, 400, true);
    add_image_size('portfolio-large', 1200, 800, true);
    add_image_size('hero-image', 800, 1000, true);
    add_image_size('post-thumb', 400, 300, true);

    // Register navigation menus
    register_nav_menus(array(
        'primary'   => __('Primary Menu', 'amir-ahmed'),
        'footer'    => __('Footer Menu', 'amir-ahmed'),
        'social'    => __('Social Menu', 'amir-ahmed'),
    ));
}
add_action('after_setup_theme', 'amir_ahmed_setup');

/**
 * Enqueue Scripts and Styles
 */
function amir_ahmed_scripts() {
    // Google Fonts - Elza (using similar available font)
    wp_enqueue_style('google-fonts', 'https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap', array(), null);

    // Main stylesheet
    wp_enqueue_style('amir-ahmed-style', get_stylesheet_uri(), array(), '1.0.0');

    // Custom CSS
    wp_enqueue_style('amir-ahmed-custom', get_template_directory_uri() . '/assets/css/custom.css', array('amir-ahmed-style'), '1.0.0');

    // Main JavaScript
    wp_enqueue_script('amir-ahmed-main', get_template_directory_uri() . '/assets/js/main.js', array('jquery'), '1.0.0', true);

    // Localize script
    wp_localize_script('amir-ahmed-main', 'amirAhmed', array(
        'ajaxurl' => admin_url('admin-ajax.php'),
        'nonce'   => wp_create_nonce('amir_ahmed_nonce'),
    ));
}
add_action('wp_enqueue_scripts', 'amir_ahmed_scripts');

/**
 * Register Custom Post Types
 */
function amir_ahmed_register_post_types() {
    // Portfolio
    register_post_type('portfolio', array(
        'labels' => array(
            'name'               => __('Portfolio', 'amir-ahmed'),
            'singular_name'      => __('Portfolio Item', 'amir-ahmed'),
            'add_new'            => __('Add New', 'amir-ahmed'),
            'add_new_item'       => __('Add New Portfolio Item', 'amir-ahmed'),
            'edit_item'          => __('Edit Portfolio Item', 'amir-ahmed'),
            'new_item'           => __('New Portfolio Item', 'amir-ahmed'),
            'view_item'          => __('View Portfolio Item', 'amir-ahmed'),
            'search_items'       => __('Search Portfolio', 'amir-ahmed'),
            'not_found'          => __('No portfolio items found', 'amir-ahmed'),
            'not_found_in_trash' => __('No portfolio items found in Trash', 'amir-ahmed'),
        ),
        'public'        => true,
        'has_archive'   => true,
        'menu_icon'     => 'dashicons-portfolio',
        'supports'      => array('title', 'editor', 'thumbnail', 'excerpt'),
        'rewrite'       => array('slug' => 'portfolio'),
        'show_in_rest'  => true,
    ));

    // Portfolio Categories
    register_taxonomy('portfolio_category', 'portfolio', array(
        'labels' => array(
            'name'          => __('Portfolio Categories', 'amir-ahmed'),
            'singular_name' => __('Portfolio Category', 'amir-ahmed'),
        ),
        'hierarchical' => true,
        'show_in_rest' => true,
        'rewrite'      => array('slug' => 'portfolio-category'),
    ));

    // Music Releases
    register_post_type('release', array(
        'labels' => array(
            'name'               => __('Releases', 'amir-ahmed'),
            'singular_name'      => __('Release', 'amir-ahmed'),
            'add_new'            => __('Add New', 'amir-ahmed'),
            'add_new_item'       => __('Add New Release', 'amir-ahmed'),
            'edit_item'          => __('Edit Release', 'amir-ahmed'),
            'new_item'           => __('New Release', 'amir-ahmed'),
            'view_item'          => __('View Release', 'amir-ahmed'),
            'search_items'       => __('Search Releases', 'amir-ahmed'),
        ),
        'public'        => true,
        'has_archive'   => true,
        'menu_icon'     => 'dashicons-format-audio',
        'supports'      => array('title', 'editor', 'thumbnail', 'excerpt', 'custom-fields'),
        'rewrite'       => array('slug' => 'releases'),
        'show_in_rest'  => true,
    ));

    // Podcast Episodes
    register_post_type('podcast', array(
        'labels' => array(
            'name'               => __('Podcast', 'amir-ahmed'),
            'singular_name'      => __('Episode', 'amir-ahmed'),
            'add_new'            => __('Add New', 'amir-ahmed'),
            'add_new_item'       => __('Add New Episode', 'amir-ahmed'),
            'edit_item'          => __('Edit Episode', 'amir-ahmed'),
            'new_item'           => __('New Episode', 'amir-ahmed'),
            'view_item'          => __('View Episode', 'amir-ahmed'),
            'search_items'       => __('Search Episodes', 'amir-ahmed'),
        ),
        'public'        => true,
        'has_archive'   => true,
        'menu_icon'     => 'dashicons-microphone',
        'supports'      => array('title', 'editor', 'thumbnail', 'excerpt', 'custom-fields'),
        'rewrite'       => array('slug' => 'podcast'),
        'show_in_rest'  => true,
    ));

    // Testimonials
    register_post_type('testimonial', array(
        'labels' => array(
            'name'               => __('Testimonials', 'amir-ahmed'),
            'singular_name'      => __('Testimonial', 'amir-ahmed'),
            'add_new'            => __('Add New', 'amir-ahmed'),
            'add_new_item'       => __('Add New Testimonial', 'amir-ahmed'),
        ),
        'public'        => false,
        'show_ui'       => true,
        'menu_icon'     => 'dashicons-format-quote',
        'supports'      => array('title', 'editor', 'thumbnail'),
        'show_in_rest'  => true,
    ));

    // Downloads
    register_post_type('download', array(
        'labels' => array(
            'name'               => __('Downloads', 'amir-ahmed'),
            'singular_name'      => __('Download', 'amir-ahmed'),
            'add_new'            => __('Add New', 'amir-ahmed'),
            'add_new_item'       => __('Add New Download', 'amir-ahmed'),
        ),
        'public'        => true,
        'has_archive'   => true,
        'menu_icon'     => 'dashicons-download',
        'supports'      => array('title', 'editor', 'thumbnail', 'custom-fields'),
        'rewrite'       => array('slug' => 'downloads'),
        'show_in_rest'  => true,
    ));
}
add_action('init', 'amir_ahmed_register_post_types');

/**
 * Register Widget Areas
 */
function amir_ahmed_widgets_init() {
    register_sidebar(array(
        'name'          => __('Footer Widget 1', 'amir-ahmed'),
        'id'            => 'footer-1',
        'description'   => __('Footer widget area 1', 'amir-ahmed'),
        'before_widget' => '<div id="%1$s" class="footer-widget %2$s">',
        'after_widget'  => '</div>',
        'before_title'  => '<h4 class="footer-title">',
        'after_title'   => '</h4>',
    ));

    register_sidebar(array(
        'name'          => __('Footer Widget 2', 'amir-ahmed'),
        'id'            => 'footer-2',
        'description'   => __('Footer widget area 2', 'amir-ahmed'),
        'before_widget' => '<div id="%1$s" class="footer-widget %2$s">',
        'after_widget'  => '</div>',
        'before_title'  => '<h4 class="footer-title">',
        'after_title'   => '</h4>',
    ));

    register_sidebar(array(
        'name'          => __('Sidebar', 'amir-ahmed'),
        'id'            => 'sidebar-1',
        'description'   => __('Main sidebar', 'amir-ahmed'),
        'before_widget' => '<div id="%1$s" class="widget %2$s">',
        'after_widget'  => '</div>',
        'before_title'  => '<h4 class="widget-title">',
        'after_title'   => '</h4>',
    ));
}
add_action('widgets_init', 'amir_ahmed_widgets_init');

/**
 * Custom Walker for Navigation Menu
 */
class Amir_Ahmed_Nav_Walker extends Walker_Nav_Menu {
    public function start_el(&$output, $item, $depth = 0, $args = null, $id = 0) {
        $classes = empty($item->classes) ? array() : (array) $item->classes;
        $class_names = join(' ', apply_filters('nav_menu_css_class', array_filter($classes), $item, $args, $depth));
        $class_names = $class_names ? ' class="' . esc_attr($class_names) . '"' : '';

        $output .= '<li' . $class_names . '>';

        $atts = array(
            'title'  => !empty($item->attr_title) ? $item->attr_title : '',
            'target' => !empty($item->target) ? $item->target : '',
            'rel'    => !empty($item->xfn) ? $item->xfn : '',
            'href'   => !empty($item->url) ? $item->url : '',
        );

        $atts = apply_filters('nav_menu_link_attributes', $atts, $item, $args, $depth);

        $attributes = '';
        foreach ($atts as $attr => $value) {
            if (!empty($value)) {
                $value = ('href' === $attr) ? esc_url($value) : esc_attr($value);
                $attributes .= ' ' . $attr . '="' . $value . '"';
            }
        }

        $title = apply_filters('the_title', $item->title, $item->ID);
        $item_output = $args->before;
        $item_output .= '<a' . $attributes . '>';
        $item_output .= $args->link_before . $title . $args->link_after;
        $item_output .= '</a>';
        $item_output .= $args->after;

        $output .= apply_filters('walker_nav_menu_start_el', $item_output, $item, $depth, $args);
    }
}

/**
 * Theme Customizer - Comprehensive Header & Footer Settings
 */
function amir_ahmed_customize_register($wp_customize) {

    // ==========================================================================
    // PANEL: Header Settings
    // ==========================================================================
    $wp_customize->add_panel('header_panel', array(
        'title'    => __('Header Settings', 'amir-ahmed'),
        'priority' => 25,
    ));

    // Section: Logo Settings
    $wp_customize->add_section('header_logo', array(
        'title' => __('Logo Settings', 'amir-ahmed'),
        'panel' => 'header_panel',
    ));

    $wp_customize->add_setting('logo_text_line1', array(
        'default'           => 'AMIR',
        'sanitize_callback' => 'sanitize_text_field',
        'transport'         => 'postMessage',
    ));

    $wp_customize->add_control('logo_text_line1', array(
        'label'   => __('Logo Text Line 1', 'amir-ahmed'),
        'section' => 'header_logo',
        'type'    => 'text',
    ));

    $wp_customize->add_setting('logo_text_line2', array(
        'default'           => 'AHMED',
        'sanitize_callback' => 'sanitize_text_field',
        'transport'         => 'postMessage',
    ));

    $wp_customize->add_control('logo_text_line2', array(
        'label'   => __('Logo Text Line 2', 'amir-ahmed'),
        'section' => 'header_logo',
        'type'    => 'text',
    ));

    $wp_customize->add_setting('show_logo_icon', array(
        'default'           => true,
        'sanitize_callback' => 'wp_validate_boolean',
    ));

    $wp_customize->add_control('show_logo_icon', array(
        'label'   => __('Show Logo Icon (Asterisk)', 'amir-ahmed'),
        'section' => 'header_logo',
        'type'    => 'checkbox',
    ));

    $wp_customize->add_setting('logo_icon_image', array(
        'sanitize_callback' => 'absint',
    ));

    $wp_customize->add_control(new WP_Customize_Media_Control($wp_customize, 'logo_icon_image', array(
        'label'       => __('Custom Logo Icon', 'amir-ahmed'),
        'description' => __('Upload custom icon or leave empty for default asterisk', 'amir-ahmed'),
        'section'     => 'header_logo',
        'mime_type'   => 'image',
    )));

    // Section: Header Style
    $wp_customize->add_section('header_style', array(
        'title' => __('Header Style', 'amir-ahmed'),
        'panel' => 'header_panel',
    ));

    $wp_customize->add_setting('header_style', array(
        'default'           => 'light',
        'sanitize_callback' => 'sanitize_text_field',
    ));

    $wp_customize->add_control('header_style', array(
        'label'   => __('Header Style', 'amir-ahmed'),
        'section' => 'header_style',
        'type'    => 'select',
        'choices' => array(
            'light'       => __('Light (White Background)', 'amir-ahmed'),
            'dark'        => __('Dark (Black Background)', 'amir-ahmed'),
            'transparent' => __('Transparent', 'amir-ahmed'),
        ),
    ));

    $wp_customize->add_setting('header_sticky', array(
        'default'           => true,
        'sanitize_callback' => 'wp_validate_boolean',
    ));

    $wp_customize->add_control('header_sticky', array(
        'label'   => __('Enable Sticky Header', 'amir-ahmed'),
        'section' => 'header_style',
        'type'    => 'checkbox',
    ));

    // Section: Header CTA Button
    $wp_customize->add_section('header_cta', array(
        'title' => __('Header CTA Button', 'amir-ahmed'),
        'panel' => 'header_panel',
    ));

    $wp_customize->add_setting('show_header_cta', array(
        'default'           => false,
        'sanitize_callback' => 'wp_validate_boolean',
    ));

    $wp_customize->add_control('show_header_cta', array(
        'label'   => __('Show CTA Button in Header', 'amir-ahmed'),
        'section' => 'header_cta',
        'type'    => 'checkbox',
    ));

    $wp_customize->add_setting('header_cta_text', array(
        'default'           => 'Get In Touch',
        'sanitize_callback' => 'sanitize_text_field',
    ));

    $wp_customize->add_control('header_cta_text', array(
        'label'   => __('CTA Button Text', 'amir-ahmed'),
        'section' => 'header_cta',
        'type'    => 'text',
    ));

    $wp_customize->add_setting('header_cta_url', array(
        'default'           => '/contact',
        'sanitize_callback' => 'esc_url_raw',
    ));

    $wp_customize->add_control('header_cta_url', array(
        'label'   => __('CTA Button URL', 'amir-ahmed'),
        'section' => 'header_cta',
        'type'    => 'url',
    ));

    // ==========================================================================
    // PANEL: Footer Settings
    // ==========================================================================
    $wp_customize->add_panel('footer_panel', array(
        'title'    => __('Footer Settings', 'amir-ahmed'),
        'priority' => 26,
    ));

    // Section: Footer Logo & About
    $wp_customize->add_section('footer_about', array(
        'title' => __('Footer About', 'amir-ahmed'),
        'panel' => 'footer_panel',
    ));

    $wp_customize->add_setting('footer_logo_text_line1', array(
        'default'           => 'AMIR',
        'sanitize_callback' => 'sanitize_text_field',
        'transport'         => 'postMessage',
    ));

    $wp_customize->add_control('footer_logo_text_line1', array(
        'label'   => __('Footer Logo Text Line 1', 'amir-ahmed'),
        'section' => 'footer_about',
        'type'    => 'text',
    ));

    $wp_customize->add_setting('footer_logo_text_line2', array(
        'default'           => 'AHMED',
        'sanitize_callback' => 'sanitize_text_field',
        'transport'         => 'postMessage',
    ));

    $wp_customize->add_control('footer_logo_text_line2', array(
        'label'   => __('Footer Logo Text Line 2', 'amir-ahmed'),
        'section' => 'footer_about',
        'type'    => 'text',
    ));

    $wp_customize->add_setting('footer_description', array(
        'default'           => 'Interdisciplinary Artist & Creative Strategist based in the United Arab Emirates. Blending art with strategy to create impactful results.',
        'sanitize_callback' => 'sanitize_textarea_field',
        'transport'         => 'postMessage',
    ));

    $wp_customize->add_control('footer_description', array(
        'label'   => __('Footer Description', 'amir-ahmed'),
        'section' => 'footer_about',
        'type'    => 'textarea',
    ));

    $wp_customize->add_setting('show_footer_social', array(
        'default'           => true,
        'sanitize_callback' => 'wp_validate_boolean',
    ));

    $wp_customize->add_control('show_footer_social', array(
        'label'   => __('Show Social Links in Footer', 'amir-ahmed'),
        'section' => 'footer_about',
        'type'    => 'checkbox',
    ));

    // Section: Footer Column 1 (Quick Links)
    $wp_customize->add_section('footer_column1', array(
        'title' => __('Footer Column 1 - Quick Links', 'amir-ahmed'),
        'panel' => 'footer_panel',
    ));

    $wp_customize->add_setting('footer_col1_title', array(
        'default'           => 'Quick Links',
        'sanitize_callback' => 'sanitize_text_field',
        'transport'         => 'postMessage',
    ));

    $wp_customize->add_control('footer_col1_title', array(
        'label'   => __('Column 1 Title', 'amir-ahmed'),
        'section' => 'footer_column1',
        'type'    => 'text',
    ));

    // Quick Links - 6 links
    for ($i = 1; $i <= 6; $i++) {
        $wp_customize->add_setting('footer_link' . $i . '_text', array(
            'default'           => '',
            'sanitize_callback' => 'sanitize_text_field',
        ));

        $wp_customize->add_control('footer_link' . $i . '_text', array(
            'label'   => sprintf(__('Link %d Text', 'amir-ahmed'), $i),
            'section' => 'footer_column1',
            'type'    => 'text',
        ));

        $wp_customize->add_setting('footer_link' . $i . '_url', array(
            'default'           => '',
            'sanitize_callback' => 'esc_url_raw',
        ));

        $wp_customize->add_control('footer_link' . $i . '_url', array(
            'label'   => sprintf(__('Link %d URL', 'amir-ahmed'), $i),
            'section' => 'footer_column1',
            'type'    => 'url',
        ));
    }

    // Section: Footer Column 2 (Services)
    $wp_customize->add_section('footer_column2', array(
        'title' => __('Footer Column 2 - Services', 'amir-ahmed'),
        'panel' => 'footer_panel',
    ));

    $wp_customize->add_setting('footer_col2_title', array(
        'default'           => 'Services',
        'sanitize_callback' => 'sanitize_text_field',
        'transport'         => 'postMessage',
    ));

    $wp_customize->add_control('footer_col2_title', array(
        'label'   => __('Column 2 Title', 'amir-ahmed'),
        'section' => 'footer_column2',
        'type'    => 'text',
    ));

    // Services Links - 6 links
    for ($i = 1; $i <= 6; $i++) {
        $wp_customize->add_setting('footer_service' . $i . '_text', array(
            'default'           => '',
            'sanitize_callback' => 'sanitize_text_field',
        ));

        $wp_customize->add_control('footer_service' . $i . '_text', array(
            'label'   => sprintf(__('Service %d Text', 'amir-ahmed'), $i),
            'section' => 'footer_column2',
            'type'    => 'text',
        ));

        $wp_customize->add_setting('footer_service' . $i . '_url', array(
            'default'           => '',
            'sanitize_callback' => 'esc_url_raw',
        ));

        $wp_customize->add_control('footer_service' . $i . '_url', array(
            'label'   => sprintf(__('Service %d URL', 'amir-ahmed'), $i),
            'section' => 'footer_column2',
            'type'    => 'url',
        ));
    }

    // Section: Footer Column 3 (Contact)
    $wp_customize->add_section('footer_column3', array(
        'title' => __('Footer Column 3 - Contact', 'amir-ahmed'),
        'panel' => 'footer_panel',
    ));

    $wp_customize->add_setting('footer_col3_title', array(
        'default'           => 'Contact',
        'sanitize_callback' => 'sanitize_text_field',
        'transport'         => 'postMessage',
    ));

    $wp_customize->add_control('footer_col3_title', array(
        'label'   => __('Column 3 Title', 'amir-ahmed'),
        'section' => 'footer_column3',
        'type'    => 'text',
    ));

    $wp_customize->add_setting('footer_email', array(
        'default'           => 'hello@amirahmed.com',
        'sanitize_callback' => 'sanitize_email',
        'transport'         => 'postMessage',
    ));

    $wp_customize->add_control('footer_email', array(
        'label'   => __('Email Address', 'amir-ahmed'),
        'section' => 'footer_column3',
        'type'    => 'email',
    ));

    $wp_customize->add_setting('footer_phone', array(
        'default'           => '',
        'sanitize_callback' => 'sanitize_text_field',
        'transport'         => 'postMessage',
    ));

    $wp_customize->add_control('footer_phone', array(
        'label'   => __('Phone Number', 'amir-ahmed'),
        'section' => 'footer_column3',
        'type'    => 'text',
    ));

    $wp_customize->add_setting('footer_location', array(
        'default'           => 'Dubai, United Arab Emirates',
        'sanitize_callback' => 'sanitize_text_field',
        'transport'         => 'postMessage',
    ));

    $wp_customize->add_control('footer_location', array(
        'label'   => __('Location', 'amir-ahmed'),
        'section' => 'footer_column3',
        'type'    => 'text',
    ));

    $wp_customize->add_setting('footer_business_hours', array(
        'default'           => 'Sun - Thu: 9:00 AM - 6:00 PM',
        'sanitize_callback' => 'sanitize_text_field',
        'transport'         => 'postMessage',
    ));

    $wp_customize->add_control('footer_business_hours', array(
        'label'   => __('Business Hours', 'amir-ahmed'),
        'section' => 'footer_column3',
        'type'    => 'text',
    ));

    // Section: Footer Bottom
    $wp_customize->add_section('footer_bottom', array(
        'title' => __('Footer Bottom', 'amir-ahmed'),
        'panel' => 'footer_panel',
    ));

    $wp_customize->add_setting('footer_copyright', array(
        'default'           => '© {year} Amir Ahmed. All rights reserved.',
        'sanitize_callback' => 'sanitize_text_field',
        'transport'         => 'postMessage',
    ));

    $wp_customize->add_control('footer_copyright', array(
        'label'       => __('Copyright Text', 'amir-ahmed'),
        'description' => __('Use {year} to display current year automatically', 'amir-ahmed'),
        'section'     => 'footer_bottom',
        'type'        => 'text',
    ));

    $wp_customize->add_setting('footer_bottom_text', array(
        'default'           => 'Designed with passion',
        'sanitize_callback' => 'sanitize_text_field',
        'transport'         => 'postMessage',
    ));

    $wp_customize->add_control('footer_bottom_text', array(
        'label'   => __('Footer Bottom Right Text', 'amir-ahmed'),
        'section' => 'footer_bottom',
        'type'    => 'text',
    ));

    $wp_customize->add_setting('show_back_to_top', array(
        'default'           => true,
        'sanitize_callback' => 'wp_validate_boolean',
    ));

    $wp_customize->add_control('show_back_to_top', array(
        'label'   => __('Show Back to Top Button', 'amir-ahmed'),
        'section' => 'footer_bottom',
        'type'    => 'checkbox',
    ));

    // ==========================================================================
    // Hero Section
    // ==========================================================================
    $wp_customize->add_section('hero_section', array(
        'title'    => __('Hero Section', 'amir-ahmed'),
        'priority' => 30,
    ));

    $wp_customize->add_setting('hero_tagline', array(
        'default'           => 'Interdisciplinary Artist',
        'sanitize_callback' => 'sanitize_text_field',
        'transport'         => 'postMessage',
    ));

    $wp_customize->add_control('hero_tagline', array(
        'label'   => __('Hero Tagline', 'amir-ahmed'),
        'section' => 'hero_section',
        'type'    => 'text',
    ));

    $wp_customize->add_setting('hero_title', array(
        'default'           => 'Interdisciplinary Artist & Creative Strategist',
        'sanitize_callback' => 'sanitize_text_field',
        'transport'         => 'postMessage',
    ));

    $wp_customize->add_control('hero_title', array(
        'label'   => __('Hero Title', 'amir-ahmed'),
        'section' => 'hero_section',
        'type'    => 'text',
    ));

    $wp_customize->add_setting('hero_description', array(
        'default'           => 'Amir Ahmed Omar Mohamed is an entrepreneur, interdisciplinary artist, and creative strategist based in the United Arab Emirates.',
        'sanitize_callback' => 'sanitize_textarea_field',
        'transport'         => 'postMessage',
    ));

    $wp_customize->add_control('hero_description', array(
        'label'   => __('Hero Description', 'amir-ahmed'),
        'section' => 'hero_section',
        'type'    => 'textarea',
    ));

    $wp_customize->add_setting('hero_image', array(
        'sanitize_callback' => 'absint',
    ));

    $wp_customize->add_control(new WP_Customize_Media_Control($wp_customize, 'hero_image', array(
        'label'     => __('Hero Image', 'amir-ahmed'),
        'section'   => 'hero_section',
        'mime_type' => 'image',
    )));

    $wp_customize->add_setting('hero_btn1_text', array(
        'default'           => 'View Portfolio',
        'sanitize_callback' => 'sanitize_text_field',
    ));

    $wp_customize->add_control('hero_btn1_text', array(
        'label'   => __('Primary Button Text', 'amir-ahmed'),
        'section' => 'hero_section',
        'type'    => 'text',
    ));

    $wp_customize->add_setting('hero_btn1_url', array(
        'default'           => '/portfolio',
        'sanitize_callback' => 'esc_url_raw',
    ));

    $wp_customize->add_control('hero_btn1_url', array(
        'label'   => __('Primary Button URL', 'amir-ahmed'),
        'section' => 'hero_section',
        'type'    => 'url',
    ));

    $wp_customize->add_setting('hero_btn2_text', array(
        'default'           => 'Get In Touch',
        'sanitize_callback' => 'sanitize_text_field',
    ));

    $wp_customize->add_control('hero_btn2_text', array(
        'label'   => __('Secondary Button Text', 'amir-ahmed'),
        'section' => 'hero_section',
        'type'    => 'text',
    ));

    $wp_customize->add_setting('hero_btn2_url', array(
        'default'           => '/contact',
        'sanitize_callback' => 'esc_url_raw',
    ));

    $wp_customize->add_control('hero_btn2_url', array(
        'label'   => __('Secondary Button URL', 'amir-ahmed'),
        'section' => 'hero_section',
        'type'    => 'url',
    ));

    // ==========================================================================
    // Contact Information
    // ==========================================================================
    $wp_customize->add_section('contact_info', array(
        'title'    => __('Contact Information', 'amir-ahmed'),
        'priority' => 35,
    ));

    $wp_customize->add_setting('contact_email', array(
        'default'           => 'hello@amirahmed.com',
        'sanitize_callback' => 'sanitize_email',
    ));

    $wp_customize->add_control('contact_email', array(
        'label'   => __('Email Address', 'amir-ahmed'),
        'section' => 'contact_info',
        'type'    => 'email',
    ));

    $wp_customize->add_setting('contact_phone', array(
        'default'           => '',
        'sanitize_callback' => 'sanitize_text_field',
    ));

    $wp_customize->add_control('contact_phone', array(
        'label'   => __('Phone Number', 'amir-ahmed'),
        'section' => 'contact_info',
        'type'    => 'text',
    ));

    $wp_customize->add_setting('contact_location', array(
        'default'           => 'Dubai, United Arab Emirates',
        'sanitize_callback' => 'sanitize_text_field',
    ));

    $wp_customize->add_control('contact_location', array(
        'label'   => __('Location', 'amir-ahmed'),
        'section' => 'contact_info',
        'type'    => 'text',
    ));

    $wp_customize->add_setting('contact_business_hours', array(
        'default'           => 'Sun - Thu: 9:00 AM - 6:00 PM',
        'sanitize_callback' => 'sanitize_text_field',
    ));

    $wp_customize->add_control('contact_business_hours', array(
        'label'   => __('Business Hours', 'amir-ahmed'),
        'section' => 'contact_info',
        'type'    => 'text',
    ));

    // ==========================================================================
    // Social Media Links
    // ==========================================================================
    $wp_customize->add_section('social_links', array(
        'title'    => __('Social Media Links', 'amir-ahmed'),
        'priority' => 40,
    ));

    $social_platforms = array(
        'instagram' => 'Instagram',
        'linkedin'  => 'LinkedIn',
        'youtube'   => 'YouTube',
        'twitter'   => 'Twitter/X',
        'spotify'   => 'Spotify',
        'facebook'  => 'Facebook',
        'tiktok'    => 'TikTok',
        'soundcloud' => 'SoundCloud',
    );

    foreach ($social_platforms as $platform => $label) {
        $wp_customize->add_setting('social_' . $platform, array(
            'default'           => '',
            'sanitize_callback' => 'esc_url_raw',
        ));

        $wp_customize->add_control('social_' . $platform, array(
            'label'   => $label . ' URL',
            'section' => 'social_links',
            'type'    => 'url',
        ));
    }
}
add_action('customize_register', 'amir_ahmed_customize_register');

/**
 * Contact Form Handler
 */
function amir_ahmed_contact_form() {
    check_ajax_referer('amir_ahmed_nonce', 'nonce');

    $name    = sanitize_text_field($_POST['name'] ?? '');
    $email   = sanitize_email($_POST['email'] ?? '');
    $subject = sanitize_text_field($_POST['subject'] ?? '');
    $message = sanitize_textarea_field($_POST['message'] ?? '');

    if (empty($name) || empty($email) || empty($message)) {
        wp_send_json_error(array('message' => 'Please fill in all required fields.'));
    }

    $to = get_option('admin_email');
    $headers = array(
        'Content-Type: text/html; charset=UTF-8',
        'From: ' . $name . ' <' . $email . '>',
        'Reply-To: ' . $email,
    );

    $email_message = sprintf(
        '<h2>New Contact Form Submission</h2>
        <p><strong>Name:</strong> %s</p>
        <p><strong>Email:</strong> %s</p>
        <p><strong>Subject:</strong> %s</p>
        <p><strong>Message:</strong></p>
        <p>%s</p>',
        esc_html($name),
        esc_html($email),
        esc_html($subject),
        nl2br(esc_html($message))
    );

    $sent = wp_mail($to, 'Contact Form: ' . $subject, $email_message, $headers);

    if ($sent) {
        wp_send_json_success(array('message' => 'Thank you! Your message has been sent.'));
    } else {
        wp_send_json_error(array('message' => 'Sorry, there was an error sending your message.'));
    }
}
add_action('wp_ajax_contact_form', 'amir_ahmed_contact_form');
add_action('wp_ajax_nopriv_contact_form', 'amir_ahmed_contact_form');

/**
 * Custom Excerpt Length
 */
function amir_ahmed_excerpt_length($length) {
    return 20;
}
add_filter('excerpt_length', 'amir_ahmed_excerpt_length');

/**
 * Custom Excerpt More
 */
function amir_ahmed_excerpt_more($more) {
    return '...';
}
add_filter('excerpt_more', 'amir_ahmed_excerpt_more');

/**
 * Add SVG Support
 */
function amir_ahmed_mime_types($mimes) {
    $mimes['svg'] = 'image/svg+xml';
    return $mimes;
}
add_filter('upload_mimes', 'amir_ahmed_mime_types');

/**
 * Disable Gutenberg for specific post types
 */
function amir_ahmed_disable_gutenberg($use_block_editor, $post_type) {
    if (in_array($post_type, array('testimonial'))) {
        return false;
    }
    return $use_block_editor;
}
add_filter('use_block_editor_for_post_type', 'amir_ahmed_disable_gutenberg', 10, 2);

/**
 * Helper function to get social links
 */
function amir_ahmed_get_social_links() {
    $platforms = array('instagram', 'linkedin', 'youtube', 'twitter', 'spotify', 'facebook', 'tiktok', 'soundcloud');
    $links = array();

    foreach ($platforms as $platform) {
        $url = get_theme_mod('social_' . $platform);
        if (!empty($url)) {
            $links[$platform] = $url;
        }
    }

    return $links;
}

/**
 * Helper function to display social links
 */
function amir_ahmed_social_links($class = 'social-links') {
    $links = amir_ahmed_get_social_links();

    if (empty($links)) {
        return;
    }

    $icons = array(
        'instagram'  => '<svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><rect x="2" y="2" width="20" height="20" rx="5" ry="5"></rect><path d="M16 11.37A4 4 0 1 1 12.63 8 4 4 0 0 1 16 11.37z"></path><line x1="17.5" y1="6.5" x2="17.51" y2="6.5"></line></svg>',
        'linkedin'   => '<svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M16 8a6 6 0 0 1 6 6v7h-4v-7a2 2 0 0 0-2-2 2 2 0 0 0-2 2v7h-4v-7a6 6 0 0 1 6-6z"></path><rect x="2" y="9" width="4" height="12"></rect><circle cx="4" cy="4" r="2"></circle></svg>',
        'youtube'    => '<svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M22.54 6.42a2.78 2.78 0 0 0-1.94-2C18.88 4 12 4 12 4s-6.88 0-8.6.46a2.78 2.78 0 0 0-1.94 2A29 29 0 0 0 1 11.75a29 29 0 0 0 .46 5.33A2.78 2.78 0 0 0 3.4 19c1.72.46 8.6.46 8.6.46s6.88 0 8.6-.46a2.78 2.78 0 0 0 1.94-2 29 29 0 0 0 .46-5.25 29 29 0 0 0-.46-5.33z"></path><polygon points="9.75 15.02 15.5 11.75 9.75 8.48 9.75 15.02"></polygon></svg>',
        'twitter'    => '<svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M23 3a10.9 10.9 0 0 1-3.14 1.53 4.48 4.48 0 0 0-7.86 3v1A10.66 10.66 0 0 1 3 4s-4 9 5 13a11.64 11.64 0 0 1-7 2c9 5 20 0 20-11.5a4.5 4.5 0 0 0-.08-.83A7.72 7.72 0 0 0 23 3z"></path></svg>',
        'spotify'    => '<svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="currentColor"><path d="M12 0C5.4 0 0 5.4 0 12s5.4 12 12 12 12-5.4 12-12S18.66 0 12 0zm5.521 17.34c-.24.359-.66.48-1.021.24-2.82-1.74-6.36-2.101-10.561-1.141-.418.122-.779-.179-.899-.539-.12-.421.18-.78.54-.9 4.56-1.021 8.52-.6 11.64 1.32.42.18.479.659.301 1.02zm1.44-3.3c-.301.42-.841.6-1.262.3-3.239-1.98-8.159-2.58-11.939-1.38-.479.12-1.02-.12-1.14-.6-.12-.48.12-1.021.6-1.141C9.6 9.9 15 10.561 18.72 12.84c.361.181.54.78.241 1.2zm.12-3.36C15.24 8.4 8.82 8.16 5.16 9.301c-.6.179-1.2-.181-1.38-.721-.18-.601.18-1.2.72-1.381 4.26-1.26 11.28-1.02 15.721 1.621.539.3.719 1.02.419 1.56-.299.421-1.02.599-1.559.3z"/></svg>',
        'facebook'   => '<svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M18 2h-3a5 5 0 0 0-5 5v3H7v4h3v8h4v-8h3l1-4h-4V7a1 1 0 0 1 1-1h3z"></path></svg>',
        'tiktok'     => '<svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="currentColor"><path d="M19.59 6.69a4.83 4.83 0 0 1-3.77-4.25V2h-3.45v13.67a2.89 2.89 0 0 1-5.2 1.74 2.89 2.89 0 0 1 2.31-4.64 2.93 2.93 0 0 1 .88.13V9.4a6.84 6.84 0 0 0-1-.05A6.33 6.33 0 0 0 5 20.1a6.34 6.34 0 0 0 10.86-4.43v-7a8.16 8.16 0 0 0 4.77 1.52v-3.4a4.85 4.85 0 0 1-1-.1z"/></svg>',
        'soundcloud' => '<svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="currentColor"><path d="M1.175 12.225c-.051 0-.094.046-.101.1l-.233 2.154.233 2.105c.007.058.05.098.101.098.05 0 .09-.04.099-.098l.255-2.105-.27-2.154c-.009-.06-.052-.1-.084-.1zm-.899 1.193c-.032 0-.058.035-.066.078l-.179 1.058.179 1.032c.008.042.034.078.066.078s.057-.036.066-.078l.204-1.032-.204-1.058c-.009-.043-.034-.078-.066-.078zm1.809-.546c-.059 0-.107.053-.115.123l-.217 1.932.217 1.853c.008.07.056.123.115.123.058 0 .106-.053.114-.123l.241-1.853-.241-1.932c-.008-.07-.056-.123-.114-.123zm.893-.337c-.066 0-.12.058-.127.131l-.201 2.027.201 1.924c.007.073.061.131.127.131.065 0 .119-.058.127-.131l.226-1.924-.226-2.027c-.008-.073-.062-.131-.127-.131zm.895-.171c-.073 0-.134.065-.141.146l-.186 2.041.186 1.939c.007.08.068.146.141.146.073 0 .133-.066.14-.146l.209-1.939-.209-2.041c-.007-.081-.067-.146-.14-.146zm.896-.17c-.08 0-.147.073-.154.161l-.171 2.041.171 1.938c.007.088.074.161.154.161.079 0 .146-.073.153-.161l.193-1.938-.193-2.041c-.007-.088-.074-.161-.153-.161zm.897-.042c-.088 0-.16.08-.168.177l-.156 1.913.156 1.91c.008.096.08.176.168.176.088 0 .159-.08.167-.176l.176-1.91-.176-1.913c-.008-.097-.079-.177-.167-.177zm.898-.084c-.095 0-.172.087-.18.191l-.14 1.784.14 1.801c.008.104.085.191.18.191.094 0 .17-.087.179-.191l.159-1.801-.159-1.784c-.009-.104-.085-.191-.179-.191zm.898.041c-.103 0-.186.094-.195.207l-.125 1.572.125 1.567c.009.113.092.206.195.206.102 0 .185-.093.193-.206l.142-1.567-.142-1.572c-.008-.113-.091-.207-.193-.207zm.898.12c-.11 0-.2.101-.208.222l-.11 1.339.11 1.335c.008.12.098.221.208.221.109 0 .199-.101.207-.221l.124-1.335-.124-1.339c-.008-.121-.098-.222-.207-.222zm5.222-.078c-.12 0-.218.109-.227.245l-.093 1.161.093 1.156c.009.136.107.245.227.245.119 0 .217-.109.225-.245l.105-1.156-.105-1.161c-.008-.136-.106-.245-.225-.245zm-4.324.106c-.117 0-.213.107-.222.24l-.094 1.133.094 1.129c.009.133.105.24.222.24.116 0 .212-.107.22-.24l.106-1.129-.106-1.133c-.008-.133-.104-.24-.22-.24zm1.125.022c-.124 0-.225.114-.233.255l-.078.906.078.901c.008.141.109.255.233.255.123 0 .224-.114.232-.255l.089-.901-.089-.906c-.008-.141-.109-.255-.232-.255zm1.116.001c-.132 0-.238.122-.247.271l-.063.678.063.674c.009.149.115.27.247.27.131 0 .237-.121.245-.27l.071-.674-.071-.678c-.008-.149-.114-.271-.245-.271zm1.116.086c-.139 0-.252.13-.26.286l-.048.449.048.445c.008.157.121.287.26.287.138 0 .251-.13.259-.287l.054-.445-.054-.449c-.008-.156-.121-.286-.259-.286zm1.117.17c-.147 0-.266.137-.274.302l-.033.22.033.216c.008.165.127.303.274.303.146 0 .265-.138.273-.303l.037-.216-.037-.22c-.008-.165-.127-.302-.273-.302zm1.117.255c-.154 0-.28.145-.288.318v.004c.008.172.134.317.288.317.153 0 .278-.145.286-.317v-.004c-.008-.173-.133-.318-.286-.318z"/></svg>',
    );

    echo '<div class="' . esc_attr($class) . '">';
    foreach ($links as $platform => $url) {
        if (isset($icons[$platform])) {
            printf(
                '<a href="%s" target="_blank" rel="noopener noreferrer" aria-label="%s">%s</a>',
                esc_url($url),
                esc_attr(ucfirst($platform)),
                $icons[$platform]
            );
        }
    }
    echo '</div>';
}

/**
 * Breadcrumb function
 */
function amir_ahmed_breadcrumb() {
    if (is_front_page()) {
        return;
    }

    echo '<nav class="breadcrumb">';
    echo '<a href="' . esc_url(home_url('/')) . '">' . __('Home', 'amir-ahmed') . '</a>';
    echo '<span>/</span>';

    if (is_single()) {
        $post_type = get_post_type();
        if ($post_type !== 'post') {
            $post_type_obj = get_post_type_object($post_type);
            echo '<a href="' . esc_url(get_post_type_archive_link($post_type)) . '">' . esc_html($post_type_obj->labels->name) . '</a>';
            echo '<span>/</span>';
        }
        echo '<span>' . esc_html(get_the_title()) . '</span>';
    } elseif (is_page()) {
        echo '<span>' . esc_html(get_the_title()) . '</span>';
    } elseif (is_archive()) {
        echo '<span>' . esc_html(get_the_archive_title()) . '</span>';
    } elseif (is_search()) {
        echo '<span>' . __('Search Results', 'amir-ahmed') . '</span>';
    }

    echo '</nav>';
}

/**
 * Add body classes
 */
function amir_ahmed_body_classes($classes) {
    if (is_front_page()) {
        $classes[] = 'home-page';
    }

    if (is_page_template('page-templates/template-dark.php')) {
        $classes[] = 'dark-theme';
    }

    return $classes;
}
add_filter('body_class', 'amir_ahmed_body_classes');

/**
 * Theme Activation - Create Pages, Menus, and Set Defaults
 */
function amir_ahmed_theme_activation() {
    // Check if setup has already been run
    if (get_option('amir_ahmed_theme_setup_complete')) {
        return;
    }

    // Pages to create
    $pages = array(
        'home' => array(
            'title'    => 'Home',
            'template' => '',
            'content'  => '',
        ),
        'about' => array(
            'title'    => 'About',
            'template' => 'page-templates/template-about.php',
            'content'  => 'Learn more about Amir Ahmed - Interdisciplinary Artist & Creative Strategist.',
        ),
        'portfolio' => array(
            'title'    => 'Portfolio',
            'template' => '',
            'content'  => 'Explore my portfolio of work across photography, videography, calligraphy, and more.',
        ),
        'press-room' => array(
            'title'    => 'Press Room',
            'template' => '',
            'content'  => 'Media coverage, press releases, and news about Amir Ahmed.',
        ),
        'releases' => array(
            'title'    => 'Releases',
            'template' => '',
            'content'  => 'Music releases, singles, and albums by Amir Ahmed.',
        ),
        'podcast' => array(
            'title'    => 'Podcast',
            'template' => '',
            'content'  => 'Listen to podcast episodes featuring conversations on creativity and strategy.',
        ),
        'downloads' => array(
            'title'    => 'Downloads',
            'template' => '',
            'content'  => 'Free digital downloads, resources, and templates.',
        ),
        'contact' => array(
            'title'    => 'Contact',
            'template' => 'page-templates/template-contact.php',
            'content'  => 'Get in touch with Amir Ahmed for collaborations, inquiries, or projects.',
        ),
    );

    $created_pages = array();

    // Create pages
    foreach ($pages as $slug => $page_data) {
        // Check if page already exists
        $existing_page = get_page_by_path($slug);

        if (!$existing_page) {
            $page_id = wp_insert_post(array(
                'post_title'     => $page_data['title'],
                'post_name'      => $slug,
                'post_content'   => $page_data['content'],
                'post_status'    => 'publish',
                'post_type'      => 'page',
                'post_author'    => 1,
                'comment_status' => 'closed',
            ));

            if ($page_id && !is_wp_error($page_id)) {
                // Set page template if specified
                if (!empty($page_data['template'])) {
                    update_post_meta($page_id, '_wp_page_template', $page_data['template']);
                }
                $created_pages[$slug] = $page_id;
            }
        } else {
            $created_pages[$slug] = $existing_page->ID;
            // Update template if needed
            if (!empty($page_data['template'])) {
                update_post_meta($existing_page->ID, '_wp_page_template', $page_data['template']);
            }
        }
    }

    // Set Home as front page
    if (isset($created_pages['home'])) {
        update_option('show_on_front', 'page');
        update_option('page_on_front', $created_pages['home']);
    }

    // Create Primary Menu
    $menu_name = 'Primary Menu';
    $menu_exists = wp_get_nav_menu_object($menu_name);

    if (!$menu_exists) {
        $menu_id = wp_create_nav_menu($menu_name);

        if (!is_wp_error($menu_id)) {
            // Menu items in order
            $menu_items = array(
                'home'       => 'Home',
                'about'      => 'About',
                'portfolio'  => 'Portfolio',
                'press-room' => 'Press Room',
                'releases'   => 'Releases',
                'podcast'    => 'Podcast',
                'contact'    => 'Contact',
            );

            $position = 0;
            foreach ($menu_items as $slug => $title) {
                if (isset($created_pages[$slug])) {
                    wp_update_nav_menu_item($menu_id, 0, array(
                        'menu-item-title'     => $title,
                        'menu-item-object'    => 'page',
                        'menu-item-object-id' => $created_pages[$slug],
                        'menu-item-type'      => 'post_type',
                        'menu-item-status'    => 'publish',
                        'menu-item-position'  => $position,
                    ));
                    $position++;
                }
            }

            // Assign menu to primary location
            $locations = get_theme_mod('nav_menu_locations');
            $locations['primary'] = $menu_id;
            set_theme_mod('nav_menu_locations', $locations);
        }
    } else {
        // Menu exists, assign to primary location
        $locations = get_theme_mod('nav_menu_locations');
        $locations['primary'] = $menu_exists->term_id;
        set_theme_mod('nav_menu_locations', $locations);
    }

    // Set default Customizer values
    $default_mods = array(
        // Header Settings
        'logo_text_line1'     => 'AMIR',
        'logo_text_line2'     => 'AHMED',
        'show_logo_icon'      => true,
        'header_style'        => 'light',
        'header_sticky'       => true,
        'show_header_cta'     => true,
        'header_cta_text'     => 'Get In Touch',
        'header_cta_url'      => '/contact',

        // Hero Section
        'hero_tagline'        => 'Interdisciplinary Artist',
        'hero_title'          => 'Interdisciplinary Artist & Creative Strategist',
        'hero_description'    => 'Amir Ahmed Omar Mohamed is an entrepreneur, interdisciplinary artist, and creative strategist based in the United Arab Emirates. Born in Dubai in 1993, he brings over 15 years of experience in visual arts and creative direction.',
        'hero_btn1_text'      => 'View Portfolio',
        'hero_btn1_url'       => '/portfolio',
        'hero_btn2_text'      => 'Get In Touch',
        'hero_btn2_url'       => '/contact',

        // Footer About
        'footer_logo_text_line1' => 'AMIR',
        'footer_logo_text_line2' => 'AHMED',
        'footer_description'     => 'Interdisciplinary Artist & Creative Strategist based in the United Arab Emirates. Blending art with strategy to create impactful results.',
        'show_footer_social'     => true,

        // Footer Column 1 - Quick Links
        'footer_col1_title'   => 'Quick Links',
        'footer_link1_text'   => 'About',
        'footer_link1_url'    => '/about',
        'footer_link2_text'   => 'Portfolio',
        'footer_link2_url'    => '/portfolio',
        'footer_link3_text'   => 'Press Room',
        'footer_link3_url'    => '/press-room',
        'footer_link4_text'   => 'Releases',
        'footer_link4_url'    => '/releases',
        'footer_link5_text'   => 'Podcast',
        'footer_link5_url'    => '/podcast',
        'footer_link6_text'   => 'Contact',
        'footer_link6_url'    => '/contact',

        // Footer Column 2 - Services
        'footer_col2_title'     => 'Services',
        'footer_service1_text'  => 'Photography',
        'footer_service1_url'   => '/portfolio/?category=photography',
        'footer_service2_text'  => 'Videography',
        'footer_service2_url'   => '/portfolio/?category=videography',
        'footer_service3_text'  => 'Calligraphy',
        'footer_service3_url'   => '/portfolio/?category=calligraphy',
        'footer_service4_text'  => 'Music Production',
        'footer_service4_url'   => '/releases',
        'footer_service5_text'  => 'Creative Direction',
        'footer_service5_url'   => '/portfolio',
        'footer_service6_text'  => 'Brand Strategy',
        'footer_service6_url'   => '/about',

        // Footer Column 3 - Contact
        'footer_col3_title'      => 'Contact',
        'footer_email'           => 'hello@amirahmed.com',
        'footer_phone'           => '+971 50 000 0000',
        'footer_location'        => 'Dubai, United Arab Emirates',
        'footer_business_hours'  => 'Sun - Thu: 9:00 AM - 6:00 PM',

        // Footer Bottom
        'footer_copyright'    => '© {year} Amir Ahmed. All rights reserved.',
        'footer_bottom_text'  => 'Designed with passion',
        'show_back_to_top'    => true,

        // Contact Information
        'contact_email'          => 'hello@amirahmed.com',
        'contact_phone'          => '+971 50 000 0000',
        'contact_location'       => 'Dubai, United Arab Emirates',
        'contact_business_hours' => 'Sun - Thu: 9:00 AM - 6:00 PM',

        // Social Media Links (placeholders - update with real links)
        'social_instagram'  => 'https://instagram.com/amirahmed',
        'social_linkedin'   => 'https://linkedin.com/in/amirahmed',
        'social_youtube'    => 'https://youtube.com/@amirahmed',
        'social_twitter'    => 'https://twitter.com/amirahmed',
        'social_spotify'    => 'https://open.spotify.com/artist/amirahmed',
    );

    // Apply default mods only if not already set
    foreach ($default_mods as $key => $value) {
        if (get_theme_mod($key) === false || get_theme_mod($key) === '') {
            set_theme_mod($key, $value);
        }
    }

    // Mark setup as complete
    update_option('amir_ahmed_theme_setup_complete', true);

    // Flush rewrite rules for custom post types
    flush_rewrite_rules();
}
add_action('after_switch_theme', 'amir_ahmed_theme_activation');

/**
 * Admin notice for theme setup
 */
function amir_ahmed_admin_notice() {
    if (!get_option('amir_ahmed_theme_setup_complete')) {
        ?>
        <div class="notice notice-info is-dismissible">
            <p><strong>Amir Ahmed Theme:</strong> Click the button below to set up your theme with default pages, menus, and settings.</p>
            <p><a href="<?php echo esc_url(admin_url('admin.php?page=amir-ahmed-setup')); ?>" class="button button-primary">Run Theme Setup</a></p>
        </div>
        <?php
    }
}
add_action('admin_notices', 'amir_ahmed_admin_notice');

/**
 * Add Theme Setup Page
 */
function amir_ahmed_add_admin_menu() {
    add_theme_page(
        'Theme Setup',
        'Theme Setup',
        'manage_options',
        'amir-ahmed-setup',
        'amir_ahmed_setup_page'
    );
}
add_action('admin_menu', 'amir_ahmed_add_admin_menu');

/**
 * Theme Setup Page Content
 */
function amir_ahmed_setup_page() {
    // Handle form submission
    if (isset($_POST['amir_ahmed_run_setup']) && check_admin_referer('amir_ahmed_setup_nonce')) {
        // Reset the setup flag to allow re-running
        delete_option('amir_ahmed_theme_setup_complete');

        // Run setup
        amir_ahmed_theme_activation();

        echo '<div class="notice notice-success"><p>Theme setup completed successfully! Pages, menus, and default settings have been configured.</p></div>';
    }

    // Handle reset
    if (isset($_POST['amir_ahmed_reset_setup']) && check_admin_referer('amir_ahmed_setup_nonce')) {
        delete_option('amir_ahmed_theme_setup_complete');
        echo '<div class="notice notice-warning"><p>Setup flag has been reset. You can now run setup again.</p></div>';
    }

    $setup_complete = get_option('amir_ahmed_theme_setup_complete');
    ?>
    <div class="wrap">
        <h1>Amir Ahmed Theme Setup</h1>

        <div style="background: #fff; padding: 20px; margin-top: 20px; border-radius: 5px; box-shadow: 0 1px 3px rgba(0,0,0,0.1);">
            <h2>Welcome to the Amir Ahmed Theme</h2>
            <p>This setup wizard will create the following:</p>
            <ul style="list-style: disc; margin-left: 20px;">
                <li><strong>Pages:</strong> Home, About, Portfolio, Press Room, Releases, Podcast, Downloads, Contact</li>
                <li><strong>Primary Navigation Menu</strong> with all pages linked</li>
                <li><strong>Homepage Settings</strong> - Home page set as front page</li>
                <li><strong>Customizer Defaults</strong> - Header, Footer, Hero, Contact, Social links</li>
            </ul>

            <?php if ($setup_complete) : ?>
                <div style="background: #d4edda; padding: 15px; border-radius: 5px; margin: 20px 0;">
                    <strong style="color: #155724;">Setup Complete!</strong>
                    <p style="color: #155724; margin: 5px 0 0;">Your theme has been configured. You can customize further in <a href="<?php echo esc_url(admin_url('customize.php')); ?>">Appearance > Customize</a>.</p>
                </div>
            <?php endif; ?>

            <form method="post" style="margin-top: 20px;">
                <?php wp_nonce_field('amir_ahmed_setup_nonce'); ?>

                <?php if (!$setup_complete) : ?>
                    <input type="submit" name="amir_ahmed_run_setup" class="button button-primary button-hero" value="Run Theme Setup">
                <?php else : ?>
                    <input type="submit" name="amir_ahmed_run_setup" class="button button-primary" value="Re-run Setup (Update Pages & Settings)">
                    <input type="submit" name="amir_ahmed_reset_setup" class="button" value="Reset Setup Flag" style="margin-left: 10px;">
                <?php endif; ?>
            </form>
        </div>

        <div style="background: #fff; padding: 20px; margin-top: 20px; border-radius: 5px; box-shadow: 0 1px 3px rgba(0,0,0,0.1);">
            <h2>Quick Links</h2>
            <ul style="list-style: none; padding: 0;">
                <li style="margin-bottom: 10px;"><a href="<?php echo esc_url(admin_url('customize.php')); ?>" class="button">Customize Theme</a></li>
                <li style="margin-bottom: 10px;"><a href="<?php echo esc_url(admin_url('nav-menus.php')); ?>" class="button">Edit Menus</a></li>
                <li style="margin-bottom: 10px;"><a href="<?php echo esc_url(admin_url('edit.php?post_type=page')); ?>" class="button">Manage Pages</a></li>
                <li style="margin-bottom: 10px;"><a href="<?php echo esc_url(admin_url('edit.php?post_type=portfolio')); ?>" class="button">Manage Portfolio</a></li>
                <li style="margin-bottom: 10px;"><a href="<?php echo esc_url(admin_url('edit.php?post_type=release')); ?>" class="button">Manage Releases</a></li>
                <li style="margin-bottom: 10px;"><a href="<?php echo esc_url(admin_url('edit.php?post_type=podcast')); ?>" class="button">Manage Podcast</a></li>
            </ul>
        </div>
    </div>
    <?php
}

/**
 * Run setup on first admin visit if not complete
 */
function amir_ahmed_maybe_run_setup() {
    if (is_admin() && current_user_can('manage_options')) {
        if (!get_option('amir_ahmed_theme_setup_complete') && !isset($_GET['page'])) {
            // Auto-run setup on first admin visit
            amir_ahmed_theme_activation();
        }
    }
}
add_action('admin_init', 'amir_ahmed_maybe_run_setup');
